 let budget = 0;
        let expenses = [];
        let chart = null;

        // Inicializar el gráfico
        function initChart() {
            const ctx = document.getElementById('budgetChart').getContext('2d');
            chart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: ['Gastado', 'Disponible'],
                    datasets: [{
                        data: [0, 100],
                        backgroundColor: [
                            'rgba(255, 107, 107, 0.8)',
                            'rgba(102, 126, 234, 0.8)'
                        ],
                        borderColor: [
                            'rgba(255, 107, 107, 1)',
                            'rgba(102, 126, 234, 1)'
                        ],
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            labels: {
                                color: 'white'
                            }
                        }
                    }
                }
            });
        }

        // Establecer presupuesto
        function setBudget(event) {
            event.preventDefault();
            const budgetInput = document.getElementById('budgetInput');
            budget = parseFloat(budgetInput.value);
            
            document.getElementById('budgetDisplay').textContent = `$${budget.toLocaleString()}`;
            updateBudgetDisplay();
            showAlert('¡Presupuesto establecido correctamente!', 'success');
        }

       // Agregar gasto
        function addExpense(event) {
            event.preventDefault();
            
            const description = document.getElementById('expenseDescription').value;
            const amount = parseFloat(document.getElementById('expenseAmount').value);
            const category = document.getElementById('expenseCategory').value;
            
            const expense = {
                id: Date.now(),
                description,
                amount,
                category,
                date: new Date().toLocaleDateString()
            };
            
            expenses.push(expense);
            updateBudgetDisplay();
            renderExpenses();
            updateAnalytics();
            
            // Limpiar el formulario
            event.target.reset();
            
            // Comprobar si se ha excedido el presupuesto
            const totalSpent = expenses.reduce((sum, exp) => sum + exp.amount, 0);
            if (totalSpent > budget && budget > 0) {
                showAlert('¡Atención! Has superado tu presupuesto mensual', 'danger');
            }
            
            showAlert('Gasto agregado correctamente', 'success');
        }

        // Actualizar la visualización del presupuesto
        function updateBudgetDisplay() {
            const totalSpent = expenses.reduce((sum, exp) => sum + exp.amount, 0);
            const remaining = budget - totalSpent;
            const percentage = budget > 0 ? (totalSpent / budget) * 100 : 0;
            
            document.getElementById('spentAmount').textContent = `$${totalSpent.toLocaleString()}`;
            document.getElementById('remainingAmount').textContent = `$${remaining.toLocaleString()}`;
            document.getElementById('progressPercentage').textContent = `${Math.round(percentage)}%`;
            
            const progressBar = document.getElementById('progressBar');
            progressBar.style.width = `${Math.min(percentage, 100)}%`;
            
            // Cambiar de color si se supera el presupuesto
            if (percentage > 100) {
                progressBar.style.background = 'linear-gradient(45deg, #ff4757, #ff3838)';
            } else if (percentage > 80) {
                progressBar.style.background = 'linear-gradient(45deg, #ffa502, #ff6348)';
            } else {
                progressBar.style.background = 'linear-gradient(45deg, #ff6b6b, #feca57)';
            }
            
            // Actualizar el gráfico
            if (chart && budget > 0) {
                chart.data.datasets[0].data = [totalSpent, Math.max(0, remaining)];
                chart.update();
            }
        }

        // Renderizas la lista de gastos
        function renderExpenses() {
            const expensesList = document.getElementById('expensesList');
            
            if (expenses.length === 0) {
                expensesList.innerHTML = `
                    <div class="text-center text-white-50 py-4">
                        <i class="fas fa-receipt fa-3x mb-3"></i>
                        <p>No hay gastos registrados aún</p>
                    </div>
                `;
                return;
            }
            
            expensesList.innerHTML = expenses.map(expense => `
                <div class="expense-item">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="text-white mb-1">${expense.description}</h6>
                            <small class="text-white-50">${expense.date}</small>
                            <span class="category-badge ms-2">${expense.category}</span>
                        </div>
                        <div class="text-end">
                            <h5 class="text-danger mb-0">-$${expense.amount.toLocaleString()}</h5>
                            <button class="btn btn-sm btn-outline-danger mt-1" onclick="deleteExpense(${expense.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `).join('');
        }

        // Eliminar gasto
        function deleteExpense(id) {

            if (confirm("Estas seguro que deseas eliminar este gasto?")){
                    expenses = expenses.filter(exp => exp.id !== id);
                    updateBudgetDisplay();
                    renderExpenses();
                    updateAnalytics();
                    showAlert('Gasto eliminado', 'info');
            }
            
        }

        // Eliminar todos los gastos
        function clearAllExpenses() {
            if (confirm('¿Estás seguro de que quieres eliminar todos los gastos?')) {
                expenses = [];
                updateBudgetDisplay();
                renderExpenses();
                updateAnalytics();
                showAlert('Todos los gastos han sido eliminados', 'info');
            }
        }

        // Actualizar analíticas
        function updateAnalytics() {
            const totalExpenses = expenses.length;
            const totalSpent = expenses.reduce((sum, exp) => sum + exp.amount, 0);
            const dailyAverage = totalExpenses > 0 ? totalSpent / 30 : 0;
            
            // Encontrar la categoría principal
            const categoryTotals = {};
            expenses.forEach(exp => {
                categoryTotals[exp.category] = (categoryTotals[exp.category] || 0) + exp.amount;
            });
            
            const topCategory = Object.keys(categoryTotals).reduce((a, b) => 
                categoryTotals[a] > categoryTotals[b] ? a : b, '-'
            );
            
            document.getElementById('totalExpenses').textContent = totalExpenses;
            document.getElementById('dailyAverage').textContent = `$${dailyAverage.toFixed(2)}`;
            document.getElementById('topCategory').textContent = topCategory;
            
            // Calcular los días que quedan en el mes
            const now = new Date();
            const daysInMonth = new Date(now.getFullYear(), now.getMonth() + 1, 0).getDate();
            const daysLeft = daysInMonth - now.getDate();
            document.getElementById('daysLeft').textContent = daysLeft;
        }

        // Muestra alertas
        function showAlert(message, type = 'info') {
            const alertArea = document.getElementById('alertArea');
            const alertClass = type === 'danger' ? 'alert-modern' : 'alert-info';
            const icon = type === 'danger' ? 'fa-exclamation-triangle' : 
                        type === 'success' ? 'fa-check-circle' : 'fa-info-circle';
            
            const alert = document.createElement('div');
            alert.className = `alert ${alertClass} alert-dismissible fade show`;
            alert.innerHTML = `
                <i class="fas ${icon} me-2"></i>
                ${message}
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="alert"></button>
            `;
            
            alertArea.appendChild(alert);
            
            // Eliminar automáticamente después de 5 segundos
            setTimeout(() => {
                if (alert.parentNode) {
                    alert.remove();
                }
            }, 5000);
        }

        // Desplazamiento suave para la navegación
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Inicializar la aplicación
        document.addEventListener('DOMContentLoaded', function() {
            initChart();
            updateAnalytics();
        });

        // Fondo de la barra de navegación al desplazarse
        window.addEventListener('scroll', () => {
            const navbar = document.querySelector('.navbar');
            if (window.scrollY > 50) {
                navbar.style.background = 'rgba(255, 255, 255, 0.15)';
            } else {
                navbar.style.background = 'rgba(255, 255, 255, 0.1)';
            }
        });